/*!
Forward models from discrete measures to observations.
*/

pub use alg_tools::linops::*;
use alg_tools::euclidean::Euclidean;
use alg_tools::error::DynError;
use alg_tools::instance::Instance;
use alg_tools::norms::{NormExponent, L2, Norm};

use crate::types::*;
use crate::measures::Radon;
pub mod sensor_grid;
pub mod bias;

/// `ForwardeModel`s are bounded preadjointable linear operators  $A ∈ 𝕃(𝒵(Ω); E)$
/// where $𝒵(Ω) ⊂ ℳ(Ω)$ is the space of sums of delta measures, presented by
/// [`crate::measures::DiscreteMeasure`], and $E$ is a [`Euclidean`] space.
pub trait ForwardModel<Domain : Space, F : Float = f64, E : NormExponent = Radon>
    : BoundedLinear<Domain, E, L2, F, Codomain=Self::Observable>
    + GEMV<F, Domain, Self::Observable>
    + Preadjointable<Domain, Self::Observable>
where
    for<'a> Self::Observable : Instance<Self::Observable>,
    Domain : Norm<F, E>,
{
    /// The codomain or value space (of “observables”) for this operator.
    /// It is assumed to be a [`Euclidean`] space, and therefore also (identified with)
    /// the domain of the preadjoint.
    type Observable : Euclidean<F, Output=Self::Observable>
                      + AXPY<F>
                      + Space
                      + Clone;

    /// Write an observable into a file.
    fn write_observable(&self, b : &Self::Observable, prefix : String) -> DynError;

    /// Returns a zero observable
    fn zero_observable(&self) -> Self::Observable;
}

/// Trait for operators $A$ for which $A_*A$ is bounded by some other operator.
pub trait AdjointProductBoundedBy<Domain : Space, D> : Linear<Domain> {
    type FloatType : Float;
    /// Return $L$ such that $A_*A ≤ LD$.
    fn adjoint_product_bound(&self, other : &D) -> Option<Self::FloatType>;
}

/// Trait for operators $A$ for which $A_*A$ is bounded by a diagonal operator.
pub trait AdjointProductPairBoundedBy<Domain : Space, D1, D2> : Linear<Domain> {
    type FloatType : Float;
    /// Return $(L, L_z)$ such that $A_*A ≤ (L_1 D_1, L_2 D_2)$.
    fn adjoint_product_pair_bound(&self, other1 : &D1, other_2 : &D2)
        -> Option<(Self::FloatType, Self::FloatType)>;
}

/// Trait for [`ForwardModel`]s whose preadjoint has Lipschitz values.
pub trait LipschitzValues {
    type FloatType : Float;
    /// Return (if one exists) a factor $L$ such that $A_*z$ is $L$-Lipschitz for all
    /// $z$ in the unit ball.
    fn value_unit_lipschitz_factor(&self) -> Option<Self::FloatType> {
        None
    }

    /// Return (if one exists) a factor $L$ such that $∇A_*z$ is $L$-Lipschitz for all
    /// $z$ in the unit ball.
    fn value_diff_unit_lipschitz_factor(&self) -> Option<Self::FloatType> {
        None
    }
}

