/*!
Implementation of scaling of functions on a manifold by a scalar.
*/

use alg_tools::mapping::Apply;
use crate::manifold::ManifoldPoint;
use crate::fb::{Grad, Desc, Prox};

/// Structure for a function of type `G`, scaled by a scalar.
pub struct Scaled<G> {
    α : f64,
    g : G,
}

impl<G> Scaled<G> {
    #[allow(dead_code)]
    pub fn new(α : f64, g : G) -> Self {
        Scaled{ α, g }
    }
}

impl<M, G : Apply<M, Output=f64>> Apply<M> for Scaled< G> {
    type Output = f64;

    fn apply(&self, x : M) -> Self::Output {
        self.g.apply(x) * self.α
    }
}

impl<M : ManifoldPoint, G : Desc<M>> Desc<M> for Scaled<G> {
    fn desc(&self, τ : f64, x : M) -> M {
        self.g.desc(τ * self.α, x)
    }
}

impl<M : ManifoldPoint, G : Grad<M>> Grad<M> for Scaled<G> {
    fn grad(&self, x : &M) -> M::Tangent {
       self.g.grad(x) * self.α
    }
}

impl<M : ManifoldPoint, G : Prox<M>> Prox<M> for Scaled<G> {
    fn prox(&self, τ : f64, x : M) -> M {
        self.g.prox(τ * self.α, x)
    }
}