/*!
Euclidean spaces.
*/

use std::ops::{Mul, MulAssign, Div, DivAssign, Add, Sub, AddAssign, SubAssign, Neg};
use crate::types::*;
use crate::mapping::Space;

/// Space (type) with a defined dot product.
///
/// `U` is the space of the multiplier, and `F` the space of scalars.
/// Since `U` ≠ `Self`, this trait can also implement dual products.
pub trait Dot<U, F> {
    fn dot(&self, other : &U) -> F;
}

/// Space (type) with Euclidean and vector space structure
///
/// The type should implement vector space operations (addition, subtraction, scalar
/// multiplication and scalar division) along with their assignment versions, as well
/// as the [`Dot`] product with respect to `Self`.
pub trait Euclidean<F : Float> : Space + Dot<Self,F>
        + Mul<F, Output=<Self as Euclidean<F>>::Output> + MulAssign<F>
        + Div<F, Output=<Self as Euclidean<F>>::Output> + DivAssign<F>
        + Add<Self, Output=<Self as Euclidean<F>>::Output>
        + Sub<Self, Output=<Self as Euclidean<F>>::Output>
        + for<'b> Add<&'b Self, Output=<Self as Euclidean<F>>::Output>
        + for<'b> Sub<&'b Self, Output=<Self as Euclidean<F>>::Output>
        + AddAssign<Self> + for<'b> AddAssign<&'b Self>
        + SubAssign<Self> + for<'b> SubAssign<&'b Self>
        + Neg<Output=<Self as Euclidean<F>>::Output> {
    type Output : Euclidean<F>;

    /// Returns origin of same dimensions as `self`.
    fn similar_origin(&self) -> <Self as Euclidean<F>>::Output;

    /// Calculate the square of the 2-norm, $\frac{1}{2}\\|x\\|_2^2$, where `self` is $x$.
    #[inline]
    fn norm2_squared(&self) -> F {
        self.dot(self)
    }

    /// Calculate the square of the 2-norm divided by 2, $\frac{1}{2}\\|x\\|_2^2$,
    /// where `self` is $x$.
    #[inline]
    fn norm2_squared_div2(&self) -> F {
        self.norm2_squared()/F::TWO
    }

    /// Calculate the 2-norm $‖x‖_2$, where `self` is $x$.
    #[inline]
    fn norm2(&self) -> F {
        self.norm2_squared().sqrt()
    }

    /// Calculate the 2-distance squared $\\|x-y\\|_2^2$, where `self` is $x$.
    fn dist2_squared(&self, y : &Self) -> F;

    /// Calculate the 2-distance $\\|x-y\\|_2$, where `self` is $x$.
    #[inline]
    fn dist2(&self, y : &Self) -> F {
        self.dist2_squared(y).sqrt()
    }

    /// Projection to the 2-ball.
    #[inline]
    fn proj_ball2(mut self, ρ : F) -> Self {
        self.proj_ball2_mut(ρ);
        self
    }

    /// In-place projection to the 2-ball.
    #[inline]
    fn proj_ball2_mut(&mut self, ρ : F) {
        let r = self.norm2();
        if r>ρ {
            *self *= ρ/r
        }
    }
}

/// Trait for [`Euclidean`] spaces with dimensions known at compile time.
pub trait StaticEuclidean<F : Float> : Euclidean<F> {
    /// Returns the origin
    fn origin() -> <Self as Euclidean<F>>::Output;
}
