use crate::types::Float;
use std::cmp::{PartialOrd,Ord,Ordering,Ordering::*};

/// A container for floating point numbers for sorting NaN as a least element
/// (to essentially ignore errorneous computations producing NaNs).
#[derive(Debug, Clone, Copy)]
pub struct NaNLeast<F : Float>(pub F);

/// Compare floating point numbers ordering nan as the least element.

impl<F : Float> Ord for NaNLeast<F> {
    #[inline]
    fn cmp(&self, NaNLeast(b) : &Self) -> Ordering {
        let NaNLeast(a) = self;
        match a.partial_cmp(b) {
            None => match (a.is_nan(), b.is_nan()) {
                (true, false) => Less,
                (false, true) => Greater,
                _ => Equal // The case (true, true) should not occur!
            }
            Some(order) => order
        }
    }
}

impl<F : Float> PartialEq for NaNLeast<F> {
    #[inline]
    fn eq(&self, other : &Self) -> bool { self.cmp(other)==Equal }
}

impl<F : Float> Eq for NaNLeast<F> { }

impl<F : Float> PartialOrd for NaNLeast<F> {
    #[inline]
    fn partial_cmp(&self, other : &Self) -> Option<Ordering> { Some(self.cmp(other)) }
}
