/*!
Some convex analysis basics
*/

use crate::mapping::{Apply, Mapping};

/// Trait for convex mappings. Has no features, just serves as a constraint
///
/// TODO: should constrain `Mapping::Codomain` to implement a partial order,
/// but this makes everything complicated with little benefit.
pub trait ConvexMapping<Domain> : Mapping<Domain> {}

/// Trait for mappings with a Fenchel conjugate
///
/// The conjugate type has to implement [`ConvexMapping`], but a `Conjugable` mapping need
/// not be convex.
pub trait Conjugable<Domain> : Mapping<Domain> {
    type DualDomain;
    type Conjugate<'a> : ConvexMapping<Self::DualDomain> where Self : 'a;

    fn conjugate(&self) -> Self::Conjugate<'_>;
}

/// Trait for mappings with a Fenchel preconjugate
///
/// In contrast to [`Conjugable`], the preconjugate need not implement [`ConvexMapping`],
/// but a `Preconjugable` mapping has be convex.
pub trait Preconjugable<Domain> : ConvexMapping<Domain> {
    type PredualDomain;
    type Preconjugate<'a> : Mapping<Self::PredualDomain> where Self : 'a;

    fn preconjugate(&self) -> Self::Preconjugate<'_>;
}

/// Trait for mappings with a proximap map
///
/// The conjugate type has to implement [`ConvexMapping`], but a `Conjugable` mapping need
/// not be convex.
pub trait HasProx<Domain> : Mapping<Domain> {
    type Prox<'a> : Mapping<Domain, Codomain=Domain> where Self : 'a;

    /// Returns a proximal mapping with weight τ
    fn prox_mapping(&self, τ : Self::Codomain) -> Self::Prox<'_>;

    /// Calculate the proximal mapping with weight τ
    fn prox(&self, z : Domain, τ : Self::Codomain) -> Domain {
        self.prox_mapping(τ).apply(z)
    }
}

