/*!
This module provides an artificial total order of floating point numbers.

The [`NaNLeast`]`<F>` container for `F` a [`Float`] puts `F` in an [`Ord`] total order with
`NaN` the least element. This allows the numbers to be sorted with `NaN` the “least important”
element when looking for the maximum. Thus erroneous computations producing `NaN` can be ignored
when there are good results.
*/

use crate::types::Float;
use std::cmp::{PartialOrd,Ord,Ordering,Ordering::*};

/// A container for floating point numbers. 
///
/// The implementation of [`Ord`] for this type type sorts `NaN` as the least element.
#[derive(Debug, Clone, Copy)]
pub struct NaNLeast<F : Float>(pub F);

impl<F : Float> Ord for NaNLeast<F> {
    #[inline]
    fn cmp(&self, NaNLeast(b) : &Self) -> Ordering {
        let NaNLeast(a) = self;
        match a.partial_cmp(b) {
            None => match (a.is_nan(), b.is_nan()) {
                (true, false) => Less,
                (false, true) => Greater,
                _ => Equal // The case (true, true) should not occur!
            }
            Some(order) => order
        }
    }
}

impl<F : Float> PartialEq for NaNLeast<F> {
    #[inline]
    fn eq(&self, other : &Self) -> bool { self.cmp(other)==Equal }
}

impl<F : Float> Eq for NaNLeast<F> { }

impl<F : Float> PartialOrd for NaNLeast<F> {
    #[inline]
    fn partial_cmp(&self, other : &Self) -> Option<Ordering> { Some(self.cmp(other)) }
}
